<?php
namespace FreeInvoice\Services;

use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

class CloudFinanceLogger implements LoggerInterface
{
    private $loggingEnabled;

    public function __construct(bool $loggingEnabled = false)
    {
        $this->loggingEnabled = $loggingEnabled;
    }

    /**
     * Metodo factory: restituisce un'istanza di CloudFinanceLogger se siamo in dev,
     * altrimenti un NullLogger che non costa risorse se disattivato in produzione.
     *
     * Assicurarsi che la costante PS_MODE_DEV sia definita (nel docker compose / Prestashop bootstrap).
     *
     * @return LoggerInterface
     */
    public static function create(): LoggerInterface
    {
        $devMode = defined('_PS_MODE_DEV_') && _PS_MODE_DEV_;
        return $devMode ? new self(true) : new NullLogger();
    }

    public function log($level, $message, array $context = []): void
    {
        if (!$this->loggingEnabled) {
            return;
        }

        if (!is_string($message)) {
            $json = json_encode($message, JSON_PRETTY_PRINT);
            if ($json === false) {
                // In caso di fallimento (ad esempio, se la struttura contiene risorse o cicli) uso print_r come fallback
                $message = print_r($message, true);
            } else {
                $message = $json;
            }
        }

        $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 3);
        if (isset($backtrace[1])) {
            $caller = $backtrace[1];
            // Ottieni solo il nome del file (senza path completo)
            $file = isset($caller['file']) ? basename($caller['file']) : 'unknown file';
            $line = isset($caller['line']) ? $caller['line'] : 'unknown line';
            // Ottieni il nome del metodo o della funzione chiamante
            $method = isset($caller['function']) ? $caller['function'] : 'unknown method';
            // Prepara il prefisso che include file, linea e metodo
            $message = "[{$file}:{$line} - {$method}] Freeinvoice: " . $message;
        }


        \PrestaShopLogger::addLog(
             $message,
            $this->mapLogLevel($level)
        );
    }

    /**
     * Mappa i livelli PSR-3 in 4 categorie:
     *  1: debug/info
     *  2: notice/warning
     *  3: error
     *  4: critical/alert/emergency
     *
     * @param string $level
     * @return int
     */
    private function mapLogLevel($level): int
    {
        if (!is_string($level)) {
            return 4; // Default a emergenza
        }
        switch ($level) {
            case 'debug':
            case 'info':
                return 1;
            case 'notice':
            case 'warning':
                return 2;
            case 'error':
            case 'critical':
            case 'alert':
                return 3;
            case 'emergency':
            default:
                return 4;
        }
    }

    // I metodi seguenti delegano internamente al metodo log()
    public function emergency($message, array $context = []): void { $this->log('emergency', $message, $context); }
    public function alert($message, array $context = []): void     { $this->log('alert', $message, $context); }
    public function critical($message, array $context = []): void  { $this->log('critical', $message, $context); }
    public function error($message, array $context = []): void     { $this->log('error', $message, $context); }
    public function warning($message, array $context = []): void   { $this->log('warning', $message, $context); }
    public function notice($message, array $context = []): void    { $this->log('notice', $message, $context); }
    public function info($message, array $context = []): void      { $this->log('info', $message, $context); }
    public function debug($message, array $context = []): void     { $this->log('debug', $message, $context); }
}
