<?php
/**
 * Modulo PrestaShop per la sincronizzazione con CloudFinance FreeInvoice
 * @author    Giuseppe Paglialonga (Cloud Finance) <g.paglialonga@cloudfinance.it>
 * @version   1.0.0
 */
declare(strict_types=1);

if (!defined('_PS_VERSION_')) {
    exit;
}
require_once _PS_MODULE_DIR_ . 'freeinvoice/services/CloudFinanceLogger.php';
require_once _PS_MODULE_DIR_ . 'freeinvoice/services/FreeInvoiceService.php';
require_once _PS_MODULE_DIR_ . 'freeinvoice/services/NotificationService.php';
require_once _PS_MODULE_DIR_ . 'freeinvoice/controllers/front/webhookreceiver.php';
require_once _PS_MODULE_DIR_ . 'freeinvoice/controllers/admin/AdminAjaxStatesController.php';

use FreeInvoice\Services\FreeInvoiceService;
use FreeInvoice\Services\CloudFinanceLogger;
use FreeInvoice\Services\NotificationService;
use Symfony\Component\Form\Extension\Core\Type\EmailType;
use Symfony\Component\Form\Extension\Core\Type\TextType;

class FreeInvoice extends Module
{

    private $log = null;
    private $freeInvoiceService = null;

    public function __construct()
    {
        $this->name = 'freeinvoice';
        $this->tab = 'administration';
        $this->version = '1.0.0';
        $this->author = 'Giuseppe Paglialonga';
        $this->need_instance = 0;
        $this->bootstrap = true;
        

        parent::__construct();

        $this->displayName = $this->l('Invia Fatture e Note di Credito a SDI');
        $this->description = $this->l('Modulo per inviare fatture e note di credito allo SDI tramite piattaforme Cloudfinance.');
        $this->ps_versions_compliancy = ['min' => '1.7.8.9', 'max' => '8.2.1'];
        $this->log = CloudFinanceLogger::create();
        $this->freeInvoiceService = new FreeInvoiceService($this->log);

    }

    public function install(): bool
    {
        if (!parent::install()) {
            return false;
        }

        $this->log->notice("Cloudfinance Install Init");
        $sqlInstallPath = __DIR__ . '/sql/install.php';

        

        if (!file_exists($sqlInstallPath)) {
            $this->log->error("Errore pathSql");
            return false;
        }

        include($sqlInstallPath);
        return
            $this->registerHook('actionPDFInvoiceRender') 
            && //Aggiunge al form di registrazione cliente i campi PEC e SDI
            $this->registerHook('additionalCustomerFormFields')
            &&
            $this->registerHook('actionCustomerAccountAdd')
            &&
            $this->registerHook('actionAfterCreateCustomerFormHandler')
            &&
            $this->registerHook('actionCustomerFormBuilderModifier')
            &&
            $this->registerHook('actionOrderSlipAdd')
            &&
            $this->registerHook('actionObjectOrderSlipAddAfter')       
            &&
            $this->installTab()
            ; 
    }

    public function uninstall(): bool
    {
        // Rimuovi le configurazioni salvate in ps_configuration
        $this->deleteConfigurationValues();

        // Deregistra gli hook
        return parent::uninstall()
            && $this->unregisterHook('actionPDFInvoiceRender')
            && $this->unregisterHook('additionalCustomerFormFields')
            && $this->unregisterHook('actionCustomerAccountAdd')
            && $this->unregisterHook('actionAfterCreateCustomerFormHandler')
            && $this->unregisterHook('actionCustomerFormBuilderModifier')
            && $this->unregisterHook('actionObjectOrderSlipAddAfter')
            && $this->uninstallTab()
            && $this->removeDatabaseChanges();
    }

    /**
     * Rimuove tutte le configurazioni salvate in `ps_configuration`
     */
    private function deleteConfigurationValues(): bool
    {
        $configKeys = [
            'FREEINVOICE_CEDENTE_EMAIL',
            'FREEINVOICE_CEDENTE_ID',
            'FREEINVOICE_CEDENTE_CODICE_FISCALE',
            'FREEINVOICE_CEDENTE_RIFERIMENTO_AMMINISTRAZIONE',
            'FREEINVOICE_CEDENTE_API_KEY',
            'FREEINVOICE_CEDENTE_TAX_REGIME',
            'FREEINVOICE_CEDENTE_PLATFORM_URL',
            'FREEINVOICE_CEDENTE_DENOMINAZIONE',
            'FREEINVOICE_CEDENTE_INDIRIZZO',
            'FREEINVOICE_CEDENTE_CIVICO',
            'FREEINVOICE_CEDENTE_CAP',
            'FREEINVOICE_CEDENTE_COMUNE',
            'FREEINVOICE_CEDENTE_PROVINCIA',
            'FREEINVOICE_CEDENTE_NAZIONE',
            'FREEINVOICE_CEDENTE_SYNC_TYPE',
            'FREEINVOICE_WEBHOOK_TOKEN',
            'FREEINVOICE_WEBHOOK_URL',
            'FREEINVOICE_GET_INVOICE_API_URL',
            'FREEINVOICE_CONFIG_SAVED',
            'FREEINVOICE_SELECTED_PLATFORM',
            'FREEINVOICE_CUSTOM_API_URL',
        ];

        $success = true;
        foreach ($configKeys as $key) {
            if (!Configuration::deleteByName($key)) {
                $success = false;
            }
        }

        return $success;
    }

    /**
     * Funzione per rimuovere la tabella e i campi aggiunti nel database
     */
    private function removeDatabaseChanges(): bool
    {
        $uninstallScriptPath = __DIR__ . '/sql/uninstall.php';
        if (file_exists($uninstallScriptPath)) {
            include($uninstallScriptPath);
            return true;
        } else {
            $this->log->error("Impossibile trovare il file uninstall.sql");
            return false;
        }
    }


    private function installTab(): bool
    {
        $tab = new Tab();
        $tab->class_name = 'AdminFreeInvoice'; // Nome del controller
        $tab->module = $this->name; // Nome del modulo
        $tab->id_parent = (int)Tab::getIdFromClassName('SELL'); // SELL è la sezione principale (es. "Vendite")
        $tab->active = 1;
        $tab->icon = 'outbox';

        foreach (Language::getLanguages(true) as $lang) {
            $tab->name[$lang['id_lang']] = $this->l('FreeInvoice');
        }

        return $tab->add();
    }

    private function uninstallTab(): bool
    {
        $idTab = (int)Tab::getIdFromClassName('AdminFreeInvoice');
        if ($idTab) {
            $tab = new Tab($idTab);
            return $tab->delete();
        }

        return true;
    }

    public function getContent(): string
    {
        $errors = [];
        $fields_value = $this->getConfigValues();
        $webhookUrl = '';
        $configPath = __DIR__ . '/config/config.json';
        if (!file_exists($configPath)) {
            $this->log->error("Errore nella lettura del config.json");
            $this->context->controller->errors[] = $this->l('Errore: il file di configurazione non è stato trovato.');
            
            return '<div class="alert alert-danger">⚠️ File <code>config.json</code> mancante. Installazione non corretta.</div>';
        }

        
        $configJson = file_get_contents($configPath);
        $configData = json_decode($configJson, true);

        if (Tools::isSubmit('submitFreeInvoiceConfig')) {
            $errors = $this->validateConfigFields();

            if (!empty($errors)) {
                foreach ($errors as $error) {
                    $this->context->controller->errors[] = $error;
                }

                // **Mantieni i valori del form in caso di errore**
                foreach ($this->getConfigFields() as $field => $options) {
                    $fields_value[$field] = Tools::getValue($field, $fields_value[$field]);
                }
            } else {

                foreach ($this->getConfigFields() as $field => $options) {
                    Configuration::updateValue($field, Tools::getValue($field));
                }
                // Ricostruzione dinamica degli endpoint
                $platform = Tools::getValue('FREEINVOICE_SELECTED_PLATFORM');
                $token = Configuration::get('FREEINVOICE_WEBHOOK_TOKEN');
                if (!$token) {
                    $token = Tools::passwdGen(32);
                    Configuration::updateValue('FREEINVOICE_WEBHOOK_TOKEN', $token);
                }
                $common = $configData['environments']['common'] ?? [];
                
                $webhookUrl = Tools::getShopDomainSsl(true, true) . $common['WEBHOOK_URL'] . '?token=' . $token;
                if ($platform === 'custom') {
                    $baseUrl = rtrim(Tools::getValue('FREEINVOICE_CUSTOM_API_URL'), '/');
                    $sendUrl = $baseUrl . '/invoices/usend';
                    $getUrl  = $baseUrl . '/invoices/{invoice_id}';
                } else {
                    $domain = rtrim($configData['environments'][$platform]['DOMAIN'] ?? '', '/');
                    $sendUrl = $domain . $common['API_URL'];
                    $getUrl  = $domain . $common['API_GET_INVOICE_URL'];
                    Configuration::deleteByName('FREEINVOICE_CUSTOM_API_URL');
                   
                }

                Configuration::updateValue('FREEINVOICE_CEDENTE_PLATFORM_URL', $sendUrl);
                Configuration::updateValue('FREEINVOICE_GET_INVOICE_API_URL', $getUrl);
                Configuration::updateValue('FREEINVOICE_WEBHOOK_URL', $webhookUrl);
                Configuration::updateValue('FREEINVOICE_CONFIG_SAVED', 1);
                
                $fields_value = $this->getConfigValues();
                $this->context->controller->confirmations[] = $this->l('Impostazioni salvate con successo.');
            }
        }

        $countries = Country::getCountries($this->context->language->id, true);
        $countryOptions = [];
        foreach ($countries as $country) {
            $countryOptions[] = [
                'id' => $country['id_country'],
                'name' => $country['name'],
                'iso_code' => $country['iso_code'],
            ];
        }

        $states = State::getStatesByIdCountry(10);
        $stateOptions = [];
        foreach ($states as $state) {
            $stateOptions[] = [
                'id' => $state['id_state'],
                'name' => $state['name'],
                'iso_code' => $state['iso_code'],
            ];
        }

        $this->context->smarty->assign([
            'action_url' => $this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&token=' . Tools::getAdminTokenLite('AdminModules'),
            'token' => Tools::getAdminTokenLite('AdminModules'),
            'regimeFiscaleOptions' => $this->getRegimeFiscaleOptions(),
            'form_fields' => $this->getConfigFields(),
            'fields_value' => $fields_value,  // Passa i valori aggiornati
            'country_options' => $countryOptions,
            'state_options' => $stateOptions,
            'ajax_url' => $this->context->link->getAdminLink('AdminAjaxStates'),
            'webhook_callback_url' => Configuration::get('FREEINVOICE_CONFIG_SAVED') ? $webhookUrl : null,
        ]);

        return $this->context->smarty->fetch($this->local_path . 'views/templates/admin/configure.tpl');
    }


    private function getRegimeFiscaleOptions(): array
    {
        return [
            ['id' => '', 'name' => 'Nessun regime fiscale'],
            ['id' => 'RF01', 'name' => 'Ordinario'],
            ['id' => 'RF02', 'name' => 'Contribuenti minimi (art.1, c.96-117, L. 244/07)'],
            ['id' => 'RF04', 'name' => 'Agricoltura e attività connesse e pesca (artt.34 e 34-bis, DPR 633/72)'],
            ['id' => 'RF05', 'name' => 'Vendita sali e tabacchi (art.74, c.1, DPR. 633/72)'],
            ['id' => 'RF06', 'name' => 'Commercio fiammiferi (art.74, c.1, DPR 633/72)'],
            ['id' => 'RF07', 'name' => 'Editoria (art.74, c.1, DPR 633/72)'],
            ['id' => 'RF08', 'name' => 'Gestione servizi telefonia pubblica (art.74, c.1, DPR 633/72)'],
            ['id' => 'RF09', 'name' => 'Rivendita documenti di trasporto pubblico e di sosta (art.74, c.1, DPR 633/72)'],
            ['id' => 'RF10', 'name' => 'Intrattenimenti, giochi e altre attività (art.74, c.6, DPR 633/72)'],
            ['id' => 'RF11', 'name' => 'Agenzie viaggi e turismo (art.74-ter, DPR 633/72)'],
            ['id' => 'RF12', 'name' => 'Agriturismo (art.5, c.2, L. 413/91)'],
            ['id' => 'RF13', 'name' => 'Vendite a domicilio (art.25-bis, c.6, DPR 600/73)'],
            ['id' => 'RF14', 'name' => 'Rivendita beni usati, oggetti d’arte, d’antiquariato o da collezione (art.36, DL 41/95)'],
            ['id' => 'RF15', 'name' => 'Agenzie di vendite all’asta di oggetti d’arte, antiquariato o da collezione (art.40-bis, DL 41/95)'],
            ['id' => 'RF16', 'name' => 'IVA per cassa P.A. (art.6, c.5, DPR 633/72)'],
            ['id' => 'RF17', 'name' => 'IVA per cassa (art. 32-bis, DL 83/2012)'],
            ['id' => 'RF18', 'name' => 'Altro'],
            ['id' => 'RF19', 'name' => 'Regime forfettario (art.1, c.54-89, L. 190/2014)'],
        ];
    }

    private function getConfigValues(): array
    {
        return [
            'FREEINVOICE_CEDENTE_EMAIL' => Configuration::get('FREEINVOICE_CEDENTE_EMAIL'),
            'FREEINVOICE_CEDENTE_ID' => Configuration::get('FREEINVOICE_CEDENTE_ID'),
            'FREEINVOICE_CEDENTE_CODICE_FISCALE' => Configuration::get('FREEINVOICE_CEDENTE_CODICE_FISCALE'),
            'FREEINVOICE_CEDENTE_RIFERIMENTO_AMMINISTRAZIONE' => Configuration::get('FREEINVOICE_CEDENTE_RIFERIMENTO_AMMINISTRAZIONE'),
            'FREEINVOICE_CEDENTE_API_KEY' => Configuration::get('FREEINVOICE_CEDENTE_API_KEY'),
            'FREEINVOICE_CEDENTE_TAX_REGIME' => Configuration::get('FREEINVOICE_CEDENTE_TAX_REGIME'),
            'FREEINVOICE_CEDENTE_DENOMINAZIONE' => Configuration::get('FREEINVOICE_CEDENTE_DENOMINAZIONE'),
            'FREEINVOICE_CEDENTE_INDIRIZZO' => Configuration::get('FREEINVOICE_CEDENTE_INDIRIZZO'),
            'FREEINVOICE_CEDENTE_CIVICO' => Configuration::get('FREEINVOICE_CEDENTE_CIVICO'),
            'FREEINVOICE_CEDENTE_CAP' => Configuration::get('FREEINVOICE_CEDENTE_CAP'),
            'FREEINVOICE_CEDENTE_COMUNE' => Configuration::get('FREEINVOICE_CEDENTE_COMUNE'),
            'FREEINVOICE_CEDENTE_PROVINCIA' => Configuration::get('FREEINVOICE_CEDENTE_PROVINCIA'),
            'FREEINVOICE_CEDENTE_NAZIONE' => Configuration::get('FREEINVOICE_CEDENTE_NAZIONE'),
            'FREEINVOICE_CEDENTE_SYNC_TYPE' => Configuration::get('FREEINVOICE_CEDENTE_SYNC_TYPE'),
            'FREEINVOICE_CONFIG_SAVED' => Configuration::get('FREEINVOICE_CONFIG_SAVED'),
            'FREEINVOICE_SELECTED_PLATFORM' => Configuration::get('FREEINVOICE_SELECTED_PLATFORM'),
            'FREEINVOICE_CUSTOM_API_URL' => Configuration::get('FREEINVOICE_CUSTOM_API_URL'),


        ];
    }
    
    private function getConfigFields(): array
    {

        return [
            'FREEINVOICE_CEDENTE_EMAIL' => ['required' => true, 'validation' => 'email'],
            'FREEINVOICE_CEDENTE_ID' => ['required' => true, 'validation' => 'partita_iva'],
            'FREEINVOICE_CEDENTE_CODICE_FISCALE' => ['required' => true, 'validation' => 'codice_fiscale'],
            'FREEINVOICE_CEDENTE_RIFERIMENTO_AMMINISTRAZIONE' => ['required' => false],
            'FREEINVOICE_CEDENTE_API_KEY' => ['required' => true],
            'FREEINVOICE_CEDENTE_DENOMINAZIONE' => ['required' => true],
            'FREEINVOICE_CEDENTE_INDIRIZZO' => ['required' => true],
            'FREEINVOICE_CEDENTE_CIVICO' => ['required' => true],
            'FREEINVOICE_CEDENTE_CAP' => ['required' => true, 'validation' => 'cap'],
            'FREEINVOICE_CEDENTE_COMUNE' => ['required' => true],
            'FREEINVOICE_CEDENTE_PROVINCIA' => ['required' => true, 'validation' => 'provincia'],
            'FREEINVOICE_CEDENTE_NAZIONE' => ['required' => true, 'validation' => 'nazione'],
            'FREEINVOICE_CEDENTE_TAX_REGIME' => ['required' => true],
            'FREEINVOICE_CEDENTE_SYNC_TYPE' => ['required' => false],
            'FREEINVOICE_SELECTED_PLATFORM' => ['required' => false],
            'FREEINVOICE_CUSTOM_API_URL' => ['required' => false],
        ];
    }

    private function validateConfigFields(): array
    {
        $errors = [];
        $fields = $this->getConfigFields();

        foreach ($fields as $field => $options) {
            $value = Tools::getValue($field);
            if ($options['required'] && empty($value)) {
                switch ($field) {
                    case 'FREEINVOICE_CEDENTE_EMAIL':
                        $errors[] = $this->l('L\'email è obbligatoria.');
                        break;
                    case 'FREEINVOICE_CEDENTE_ID':
                        $errors[] = $this->l('La Partita Iva è obbligatorio.');
                        break;
                    case 'FREEINVOICE_CEDENTE_CODICE_FISCALE':
                        $errors[] = $this->l('Il codice fiscale è obbligatorio.');
                        break;
                    case 'FREEINVOICE_CEDENTE_API_KEY':
                        $errors[] = $this->l('La chiave API è obbligatoria.');
                        break;
                    case 'FREEINVOICE_CEDENTE_DENOMINAZIONE':
                        $errors[] = $this->l('La denominazione è obbligatoria.');
                        break;
                    case 'FREEINVOICE_CEDENTE_INDIRIZZO':
                        $errors[] = $this->l('L\'indirizzo è obbligatorio.');
                        break;
                    case 'FREEINVOICE_CEDENTE_CIVICO':
                        $errors[] = $this->l('Il numero civico è obbligatorio.');
                        break;
                    case 'FREEINVOICE_CEDENTE_CAP':
                        $errors[] = $this->l('Il CAP è obbligatorio.');
                        break;
                    case 'FREEINVOICE_CEDENTE_COMUNE':
                        $errors[] = $this->l('Il comune è obbligatorio.');
                        break;
                    case 'FREEINVOICE_CEDENTE_PROVINCIA':
                        $errors[] = $this->l('La provincia è obbligatoria.');
                        break;
                    case 'FREEINVOICE_CEDENTE_NAZIONE':
                        $errors[] = $this->l('La nazione è obbligatoria.');
                        break;
                    case 'FREEINVOICE_CEDENTE_TAX_REGIME':
                        $errors[] = $this->l('La selezione del Regime Fiscale è obbligatoria.');
                        break;
                    default:
                        $errors[] = $this->l('Il campo ' . $field . 'è obbligatorio.');
                        break;
                }
            }
            // Validazioni specifiche per alcuni campi
            if (!empty($options['validation']) && !empty($value)) {
                switch ($options['validation']) {
                    case 'email':
                        if (!filter_var($value, FILTER_VALIDATE_EMAIL)) {
                            $errors[] = $this->l('Email non valida');
                        }
                        break;

                    case 'cap':
                        if (!preg_match('/^\d{5}$/', $value)) {
                            $errors[] = $this->l('CAP non valido');
                        }
                        break;

                    case 'codice_fiscale':
                        if (!preg_match('/^[A-Z0-9]{16}$/i', $value) && !preg_match('/^\d{11}$/',trim($value))) {
                            $errors[] = $this->l('Codice Fiscale non valido');
                        }
                        break;

                    case 'provincia':
                        if (!preg_match('/^[A-Z]{2}$/', $value)) {
                            $errors[] = $this->l('Codice Provincia non valido');
                        }
                        break;

                    case 'nazione':
                        if (!preg_match('/^[A-Z]{2}$/', $value)) {
                            $errors[] = $this->l('Codice Nazione non valido');
                        }
                        break;
                    case 'partita_iva': 
                        if (!preg_match('/^\d{11}$/',trim($value))) {
                            $errors[] = $this->l('Partita IVA non valida (es. 01589730629)');
                        }
                        break;
                }
            }
        }

        return $errors;
    }

    ///////HOOK///////

    public function hookActionPDFInvoiceRender($params): void
    {

        if (isset($_SERVER['REQUEST_URI']) && strpos($_SERVER['REQUEST_URI'], 'generate-invoice-pdf') !== false) {
            return;
        }
        $orderInvoiceList = $params['order_invoice_list'];
        
        foreach ($orderInvoiceList as $order_invoice) {
            $this->log->info('Hook ActionPdfInvoiceRender INIT - Fattura #' . $order_invoice->number);
            
            // Invia la fattura e riceve la risposta
            $result = $this->freeInvoiceService->sendInvoice($order_invoice, false, false);

            if (array_key_exists('success', $result) && $result['success']) {
                $message_txt = "Fattura #" . $order_invoice->number . " inviata con successo.";
                $type = "success";
                $this->get('session')->getFlashBag()->add($type, 'FreeInvoice: ' . $message_txt);
            } 
            
            if ((array_key_exists('error', $result) && $result['error']) || array_key_exists('warning', $result) && $result['warning']) {
                // Personalizza il messaggio d'errore
                $errorDetails = $result['message'];
                $errorMsg = "Errore Invio Fattura #" . $order_invoice->number . ": " . $errorDetails;

                // Notifica l'errore all'utente nel backoffice
                $this->get('session')->getFlashBag()->add('error', 'FreeInvoice: ' . $errorMsg);
                $this->log->error($errorMsg);
            }

        }
    }
     
    public function hookActionOrderSlipAdd($params){
        $this->log->error("hookActionOrderSlipAdd");

        if ($debug = $this->context->cookie->__get('last_order_slip')) {
            unset($this->context->cookie->last_order_slip);
        
            $lastOrderSlip = json_decode($debug, true);
        }

        if (isset($lastOrderSlip['id_order_slip']))
        {
            $idOrderSlip = (int)$lastOrderSlip['id_order_slip'];
        }
        if ($idOrderSlip <= 0 ){
            $this->log->error("HookActionOrderSlipAdd non ha ricevuto idOrderSlip. Contattare assistenza tecnica");
            $this->get('session')->getFlashBag()->add('error', 'FreeInvoice: HookActionOrderSlipAdd non ha ricevuto idOrderSlip. Contattare assistenza tecnica');
            return;
        }
        $orderSlip = new OrderSlip((int)$lastOrderSlip['id_order_slip']);
        $result =  $this->freeInvoiceService->sendInvoice($orderSlip, false);

        if (array_key_exists('success', $result) && $result['success']) {
            $message_txt = "Nota di Credito " . $this->freeInvoiceService->getOrderSlipNumberFormatted($orderSlip ) . " inviata con successo.";
            $this->get('session')->getFlashBag()->add('success',  'FreeInvoice: ' . $message_txt);
            $this->log->info($message_txt);
        } 
        
        if ((array_key_exists('error' , $result ) && $result['error']) || array_key_exists('warning', $result) && $result['warning']) {
            $errorMsg = "Errore invio Nota di Credito #" . $this->freeInvoiceService->getOrderSlipNumberFormatted($orderSlip ) . ". " . $result['message'] . "";
            $this->get('session')->getFlashBag()->add('error', 'FreeInvoice: ' . $errorMsg);
            $this->log->error($errorMsg);
        }
    }
    
    public function hookActionObjectOrderSlipAddAfter($params)
    {
        $orderSlip = $params['object'];
        if (!Validate::isLoadedObject($orderSlip)) {
            return;
        }
        $this->log->error("hookActionObjectOrderSlipAddAfter");
        $this->context->cookie->__set('last_order_slip', json_encode([
            'id_order_slip' => $orderSlip->id,
        ]));
        
    }

    public function hookActionCustomerAccountAdd($params)
    {
        $id = (int)$params['newCustomer']->id;
        $this->writeExtraCustomerValuesForFrontend($id);
    }


    public function hookAdditionalCustomerFormFields($params)
    {
        $extra_fields = $this->readExtraCustomerValues();

        $pec_email_value = Tools::getValue('pec_email', $extra_fields['pec_email'] ?? '');
        $sdi_code_value = Tools::getValue('sdi_code', $extra_fields['sdi_code'] ?? '');

        $extra_fields = [];

        $extra_fields[] = (new FormField)
            ->setName('pec_email')
            ->setType('text')
            ->setValue($pec_email_value)
            ->setLabel($this->l('PEC'))
            ->setConstraints(['isEmail'])
            ->setAvailableValues(['comment' => $this->l('Inserisci il tuo Indirizzo Email PEC')]);

        $extra_fields[] = (new FormField)
            ->setName('sdi_code')
            ->setType('text')
            ->setValue($sdi_code_value)
            ->setLabel($this->l('Codice Univoco'))
            ->setMaxLength(7)
            ->setAvailableValues(['comment' => $this->l('Inserisci il Codice SDI')]);

        return $extra_fields;
    }

    
    public function hookActionCustomerFormBuilderModifier($params)
    {
        $this->log->notice('Hook actionCustomerFormBuilderModifier init...');
        $formBuilder = $params['form_builder'];
        $route = $params['route'];
    
        $allFields = $formBuilder->all();
        
        $this->log->notice('Modifico il form cliente per il route: ' . $route);
        
        $pec_email_value = "";
        $sdi_code_value = "";
        
        if ($route == "admin_customers_edit") {
            $extra_fields = $this->readExtraCustomerValues($params['id']);
            $pec_email_value = $extra_fields['pec_email'];
            $sdi_code_value = $extra_fields['sdi_code'];
        }
        foreach ($allFields as $inputField => $input) {
            $formBuilder->add($input);
            
            if ($inputField == 'email') {
                $formBuilder->add(
                    'pec_email',
                    EmailType::class,
                    [
                        'label' => 'E-mail PEC',
                        'required' => false,
                        'data' => $pec_email_value
                    ]
                );
                
                $formBuilder->add(
                    'sdi_code',
                    TextType::class,
                    [
                        'label' => 'Codice SDI',
                        'required' => false,
                        'data' => $sdi_code_value,
                        'attr' => ['maxlength' => 7],
                    ]
                );
            }
        }
    }

    public function hookAddWebserviceResources($params)
    {
        $this->log->notice('hookAddWebserviceResource init...');
        return [
            'webhookreceiver' => [
                'description' => 'Gestione Webhook Personalizzata',
                'class' => 'FreeInvoiceWebhookReceiverModuleFrontController',
                'forbidden_method' => ['DELETE']
            ],
        ];
    }

    public function hookActionAfterCreateCustomerFormHandler($params)
    {
        $this->writeExtraCustomerValuesForBackend($params['id'], $params['form_data']);
    }

    //METODI AUSILIARI HOOK

    protected function writeExtraCustomerValuesForBackend($id_customer, $form_data)
    {
        $certified_email_value = $form_data['pec_email'];
        $sdi_code_value = $form_data['sdi_code'];
        
        $result = $this->writeExtraCustomerValues($id_customer, $certified_email_value, $sdi_code_value);
    }

    protected function writeExtraCustomerValuesForFrontend($id_customer)
    {
        $certified_email_value = Tools::getValue('pec_email');
        $sdi_code_value = Tools::getValue('sdi_code');
        
        $result = $this->writeExtraCustomerValues($id_customer, $certified_email_value, $sdi_code_value);
    }

    protected function writeExtraCustomerValues($id_customer, $pec_mail, $sdi_code)
    {
        $query = 'UPDATE `'._DB_PREFIX_.'customer` '
            .' SET `pec_email` = "'.pSQL($pec_mail).'", `sdi_code` = "'.pSQL($sdi_code).'"'
            .' WHERE id_customer = '.(int)$id_customer;
            $this->log->notice($query);
        Db::getInstance()->execute($query);
    }


    protected function readExtraCustomerValues($id_customer = null)
    {
        if ($id_customer == null) {
            $id_customer = Context::getContext()->customer->id;
        }
        $query = 'SELECT c.`pec_email`, c.`sdi_code`'
            .' FROM `'. _DB_PREFIX_.'customer` c '
            .' WHERE c.id_customer = '.(int)$id_customer;
            
        return  Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($query);
    }

}
