<?php

namespace FreeInvoice\Services;
use Order;
use OrderSlip;
use OrderInvoice;
use Context;
use Address;
use Country;
use Configuration;
use Customer;
use State;
use Zone;
use PrestaShop\PrestaShop\Adapter\Validate;
use Db;
//require_once _PS_MODULE_DIR_ . '../prestashop_core/Configuration.php';
class FreeInvoiceService
{
    /**
     * Genera l'XML della fattura.
     *
     * @param OrderInvoice $invoiceOrder: contiene i dati della fattura generata da prestashop
     * @return string
     */
    protected $apiKey;
    protected $cedenteIdFiscale;
    protected $codiceFiscaleCedente;
    protected $emailCedente;
    protected $codiceDestinatarioCedente;
    protected $denominazioneCedente;
    protected $indirizzoCedente;
    protected $numeroCivicoCedente;
    protected $capCedente;
    protected $comuneCedente;
    protected $provinciaCedente;
    protected $nazioneCedente;
    protected $regimeFiscaleCedente;
    protected $riferimentoAmministrazioneCedente;
    protected $platformUrl;
    protected $apiGetInvoiceUrl;
    protected $automaticSyncCedente;
    protected $log=null;

    public function __construct($logger) {
        // Tabella PS_CONFIGURATION
        $this->emailCedente = Configuration::get('FREEINVOICE_CEDENTE_EMAIL');
        $this->apiKey = Configuration::get('FREEINVOICE_CEDENTE_API_KEY');
        $this->cedenteIdFiscale = Configuration::get('FREEINVOICE_CEDENTE_ID');
        $this->codiceFiscaleCedente = Configuration::get('FREEINVOICE_CEDENTE_CODICE_FISCALE');
        $this->riferimentoAmministrazioneCedente = Configuration::get('FREEINVOICE_CEDENTE_RIFERIMENTO_AMMINISTRAZIONE');
        $this->denominazioneCedente = Configuration::get('FREEINVOICE_CEDENTE_DENOMINAZIONE');
        $this->indirizzoCedente = Configuration::get('FREEINVOICE_CEDENTE_INDIRIZZO');
        $this->numeroCivicoCedente = Configuration::get('FREEINVOICE_CEDENTE_CIVICO');
        $this->capCedente = Configuration::get('FREEINVOICE_CEDENTE_CAP');
        $this->comuneCedente = Configuration::get('FREEINVOICE_CEDENTE_COMUNE');
        $this->provinciaCedente = Configuration::get('FREEINVOICE_CEDENTE_PROVINCIA');
        $this->nazioneCedente = Configuration::get('FREEINVOICE_CEDENTE_NAZIONE');
        $this->regimeFiscaleCedente = Configuration::get('FREEINVOICE_CEDENTE_TAX_REGIME');
        $this->platformUrl = Configuration::get('FREEINVOICE_CEDENTE_PLATFORM_URL');
        $this->automaticSyncCedente = Configuration::get('FREEINVOICE_CEDENTE_SYNC_TYPE');
        $this->apiGetInvoiceUrl = Configuration::get('FREEINVOICE_GET_INVOICE_API_URL');
        $this->log = $logger;
    }

    public function generateInvoiceXML($invoiceOrder)
    { 
        if (!Validate::isLoadedObject($invoiceOrder)){
            $this->log->error('OrderInvoice / OrderSlip è un oggetto nullo');
            return ['type' => 'error', 'message' => $messageInit . 'OrderInvoice / OrderSlip è un oggetto nullo' . $invoiceOrder->id_order];
        }

        $isCreditNote = false;
        $messageInit = 'Generazione XML Fattura: ';
        
        
        if($invoiceOrder instanceof OrderSlip){
            $isCreditNote = true;
            $messageInit = 'Generazione XML Nota di Credito: ';         
        }
        
        $order = new Order($invoiceOrder->id_order);
        if (!Validate::isLoadedObject($order)){
            $this->log->error('Ordine è un oggetto nullo');
            return ['type' => 'error', 'message' => $messageInit . 'ordine è un oggetto nullo, id ' . $invoiceOrder->id_order];
        }
        
        $customer = new Customer($order->id_customer); //Cliente
        if (!Validate::isLoadedObject($customer)) {
            $this->log->error('Cliente è un oggetto nullo');
            return ['type' => 'error', 'message' => $messageInit . 'cliente è un oggetto nullo'];
        }

        $customerExtra = $this->readExtraCustomerValues($customer->id);
        $customerPecEmail = $customerExtra['pec_email'] ?? '';
        $customerSdiCode = $customerExtra['sdi_code'] ?? '';

        $this->log->notice($customer);
        $products = $invoiceOrder->getProducts(); //Prodotti Fattura/Nota di Credito
        if (!is_array($products) || empty($products)) {
            $this->log->error('Prodotti: array vuoto o non definito.');
            return ['type' => 'error', 'message' => $messageInit . 'non sono stati trovati prodotti relativi all\'ordine'];
        }
        
        $address = new Address($order->id_address_invoice); //Indirizzo di Fatturazione
        if(!Validate::isLoadedObject($address)){
            $this->log->error('Indirizzo è un oggetto nullo');
            return ['type' => 'error', 'message' => $messageInit . 'non è stato trovato un\'indirizzo di fatturazione valido'];
        }

        $idCountry = $address->id_country;
        $countryInvoiceAddress = new Country($idCountry); //Stato di fatturazione
        if(!Validate::isLoadedObject($countryInvoiceAddress)){
            $this->log->error('Stato è un oggetto nullo');
            return ['type' => 'error', 'message' => $messageInit . 'nazione è un oggetto nullo'];
        }

        $state = new State($address->id_state); //Provincia di fatturazione
        if(!Validate::isLoadedObject($state) && strcasecmp(trim($countryInvoiceAddress->iso_code), 'IT') === 0 && Country::containsStates($idCountry)){
            $this->log->error('Provincia è un oggetto nullo');
            return ['type' => 'error', 'message' => $messageInit . 'provincia è un oggetto nullo'];
        }
        $message = "";
        $b2b = true;
        $b2c = true;
        if((empty($address->vat_number) || trim($address->vat_number) === '') && strcasecmp(trim($countryInvoiceAddress->iso_code), 'IT') === 0 ){
            $b2b = false;
            $message = 'Cliente senza Partita Iva. ';
            $this->log->notice($message);
        }
        
        
        if (strcasecmp(trim($countryInvoiceAddress->iso_code), 'IT') === 0 && (empty($address->dni) || trim($address->dni) === '') && (empty($customer->siret) || trim($customer->siret) === '')) { 
            $b2c = false;
            $message = $message . 'Cliente B2C IT Codice Fiscale. ';
            $this->log->notice('Cliente senza Codice Fiscale. ');
        }
        if(!$b2b && !$b2c){
            return ['type' => 'warning', 'message' => $messageInit . '' . $message . 'La fattura non verrà inviata'];
        }

        // Inizializza l'XML con il namespace solo sul nodo radice
        $dom = new \DOMDocument('1.0', 'UTF-8');
        $root = $dom->createElementNS('http://ivaservizi.agenziaentrate.gov.it/docs/xsd/fatture/v1.2', 'p:FatturaElettronica');
        $root->setAttribute('versione', 'FPR12');
        $dom->appendChild($root);

        $context = Context::getContext();
        $this->getHeaderXML($dom, $root, $address,$customer, $customerSdiCode, $customerPecEmail, $countryInvoiceAddress);
        $this->getBodyXML($dom, $root, $order, $invoiceOrder, $products, $context, $countryInvoiceAddress, $isCreditNote);
        
        $dom->formatOutput = false;

        $this->log->notice('Fattura generata correttamente');
        return ['type' => 'success', 'xml' => $dom->saveXML()];
    }

    private function getHeaderXML ($dom, $root, $address,$customer, $customerSdiCode, $customerPecEmail, $country) 
    {
        $header = $dom->createElement('FatturaElettronicaHeader');
        $root->appendChild($header);

        $header->appendChild($this->getDatiTrasmissioneXML($dom, $customerSdiCode, $customerPecEmail, $country));
        
        $header->appendChild($this->getCedentePrestatoreXML($dom));

        $header->appendChild($this->getCessionarioCommittente($dom, $address, $customer, $country));
        
        $header->appendChild($this->getTerzoIntermediario($dom));
        
        $header->appendChild($this->getSoggettoEmittente($dom));

    }

    private function getBodyXML($dom, $root, $order, $invoiceOrder, $products, $context, $country, $isCreditNote)
    {
        // Corpo della Fattura
        $body = $dom->createElement('FatturaElettronicaBody');
        $root->appendChild($body);

        $formattedDate = (new \DateTime($order->invoice_date))->format('Y-m-d');
        $formattedDateIT = (new \DateTime($order->invoice_date))->format('d/m/Y');
        // Dati Generali
        $datiGenerali = $dom->createElement('DatiGenerali');
        $body->appendChild($datiGenerali);
        $datiGeneraliDocumento = $dom->createElement('DatiGeneraliDocumento');
        $datiGenerali->appendChild($datiGeneraliDocumento);
        //Se è una nota di credito ha tipo TD04 e la causale è "Ndc riferita a fattura n xxx del xx-xx-xxxx"
        if($isCreditNote){
            $datiGeneraliDocumento->appendChild($dom->createElement('TipoDocumento', 'TD04'));
        } else {
            $datiGeneraliDocumento->appendChild($dom->createElement('TipoDocumento', 'TD01'));
        }
        $datiGeneraliDocumento->appendChild($dom->createElement('Divisa', 'EUR'));
        $datiGeneraliDocumento->appendChild($dom->createElement('Data', $formattedDate));
        if ($isCreditNote){
            $datiGeneraliDocumento->appendChild($dom->createElement('Numero', $this->getOrderSlipNumberFormatted($invoiceOrder)));
            
            $datiGeneraliDocumento->appendChild($dom->createElement('ImportoTotaleDocumento', number_format($invoiceOrder->total_products_tax_incl + $invoiceOrder->total_shipping_tax_incl ?? 0, 2, '.', '')));
        } else {
            $datiGeneraliDocumento->appendChild($dom->createElement('Numero', $invoiceOrder->getInvoiceNumberFormatted($context->language->id)));
            $datiGeneraliDocumento->appendChild($dom->createElement('ImportoTotaleDocumento', number_format($invoiceOrder->total_paid_tax_incl ?? 0, 2, '.', '')));
        }
        
        $datiGeneraliDocumento->appendChild($dom->createElement('Arrotondamento', '0.00'));
        if($isCreditNote){
            $datiGeneraliDocumento->appendChild($dom->createElement('Causale', 'Ndc ' . $order->reference . ' del ' . $formattedDateIT));
        } else {
            $datiGeneraliDocumento->appendChild($dom->createElement('Causale', 'Riferimento all\'ordine: ' . $order->reference . ' del ' . $formattedDateIT));   
        }

        // Dati Beni Servizi
        $datiBeniServizi = $dom->createElement('DatiBeniServizi');
        $body->appendChild($datiBeniServizi);
        $counter = 0;
        foreach ($products as $index => $product) {
            $taxRate = (float)$product['tax_rate'];
            $dettaglioLinea = $dom->createElement('DettaglioLinee');
            $datiBeniServizi->appendChild($dettaglioLinea);

            $dettaglioLinea->appendChild($dom->createElement('NumeroLinea', ++$counter));
            $codiceArticolo = $dom->createElement('CodiceArticolo');
            $dettaglioLinea->appendChild($codiceArticolo);
            $codiceArticolo->appendChild($dom->createElement('CodiceTipo', 'COD'));
            $codiceArticolo->appendChild($dom->createElement('CodiceValore', $product['product_reference']));
            $dettaglioLinea->appendChild($dom->createElement('Descrizione', $product['product_name']));
            $dettaglioLinea->appendChild($dom->createElement('Quantita', number_format($product['product_quantity'], 8)));
            $dettaglioLinea->appendChild($dom->createElement('PrezzoUnitario', number_format($product['product_price'], 8)));
            $dettaglioLinea->appendChild($dom->createElement('PrezzoTotale', number_format($product['product_quantity'] * $product['product_price'], 2)));
            $dettaglioLinea->appendChild($dom->createElement('AliquotaIVA', number_format($product['tax_rate'], 2, '.', '')));
            if ($taxRate == 0) {
                switch($country->iso_code) {
                    case 'SM':
                        $dettaglioLinea->appendChild($dettaglioLinea->appendChild($dom->createElement('Natura', 'N3.3')));
                        break;
                    default:
                        $dettaglioLinea->appendChild($dettaglioLinea->appendChild($dom->createElement('Natura', 'N2.1')));
                        break;
                }
            }
        }

        $aliquote = [];
        if ($invoiceOrder->total_shipping_tax_excl > 0  || $invoiceOrder->total_shipping_tax_incl >0 ) { 
            $taxRate =number_format(((($invoiceOrder->total_shipping_tax_incl / $invoiceOrder->total_shipping_tax_excl) - 1 ) * 100), 2, '.', ''); 
            if (!isset($aliquote[(int)$taxRate])) {
                $aliquote[(int)$taxRate] = [
                    'imponibile' => 0,
                    'imposta' => 0,
                    'natura' => null // Inizializza con null per verificare successivamente
                ];
            }
            
            $dettaglioLinea = $dom->createElement('DettaglioLinee');
            $datiBeniServizi->appendChild($dettaglioLinea);
            $dettaglioLinea->appendChild($dom->createElement('NumeroLinea', ++$counter));
            $dettaglioLinea->appendChild($dom->createElement('Descrizione', "Spese di spedizione"));
            $dettaglioLinea->appendChild($dom->createElement('Quantita', "1.00000000"));
            $dettaglioLinea->appendChild($dom->createElement('PrezzoUnitario', $invoiceOrder->total_shipping_tax_excl));
            $dettaglioLinea->appendChild($dom->createElement('PrezzoTotale', $invoiceOrder->total_shipping_tax_excl));
            $dettaglioLinea->appendChild($dom->createElement('AliquotaIVA', number_format(((($invoiceOrder->total_shipping_tax_incl / $invoiceOrder->total_shipping_tax_excl) - 1 ) * 100), 2, '.', '')));
            $aliquote[(int)$taxRate]['imponibile'] += $invoiceOrder->total_shipping_tax_excl;
            $aliquote[(int)$taxRate]['imposta'] += $invoiceOrder->total_shipping_tax_incl - $invoiceOrder->total_shipping_tax_excl;

            if (isset($country->iso_code) && strcasecmp(trim($country->iso_code), 'SM') === 0 && $taxRate == 0) {
                $dettaglioLinea->appendChild($dom->createElement('Natura', 'N3.3'));
                $aliquote[(int)$taxRate]['natura'] = 'N3.3';
            }
            else if (isset($country->iso_code) && strcasecmp(trim($country->iso_code), 'IT') === 0 && $taxRate == 0) {
                $dettaglioLinea->appendChild($dom->createElement('Natura', 'N2.1'));
                $aliquote[(int)$taxRate]['natura'] = 'N2.1';
            }
            else if ($taxRate == 0){
                $dettaglioLinea->appendChild($dom->createElement('Natura', 'N2.1'));
                $aliquote[(int)$taxRate]['natura'] = 'N2.1';
            }
        }

        // Dati Riepilogo per Aliquote IVA
        foreach ($products as $product) {
            $taxRate = (float)$product['tax_rate'];
            $imponibile = $product['product_price'] * $product['product_quantity'];
            $imposta = $imponibile * ($taxRate / 100);

            if (!isset($aliquote[$taxRate])) {
                $aliquote[$taxRate] = [
                    'imponibile' => 0,
                    'imposta' => 0,
                    'natura' => null // Inizializza con null per verificare successivamente
                ];
            }

            $aliquote[$taxRate]['imponibile'] += $imponibile;
            $aliquote[$taxRate]['imposta'] += $imposta;
            
            if (!isset($aliquote[$taxRate])) {
                $aliquote[$taxRate] = ['imponibile' => 0, 'imposta' => 0];   
            }

            if ($taxRate == 0) {
                switch($country->iso_code) {
                    case 'SM':
                        $aliquote[$taxRate]['natura'] = 'N3.3';
                        break;
                    default:
                        $aliquote[$taxRate]['natura'] = 'N2.1';
                        break;
                }
            }
        }
       
        foreach ($aliquote as $aliquota => $dati) {
            $datiRiepilogo = $dom->createElement('DatiRiepilogo');
            $datiBeniServizi->appendChild($datiRiepilogo);
            
            $datiRiepilogo->appendChild($dom->createElement('AliquotaIVA', number_format($aliquota, 2, '.', '')));
            //Se l'aliquota è 0, bisogna specificare Natura (2.2.1.16), altrimenti Imposta (2.2.1.14)
            if ($aliquota == 0) {
                $datiRiepilogo->appendChild($dom->createElement('Natura', $dati['natura']));
            } 

            $datiRiepilogo->appendChild($dom->createElement('SpeseAccessorie', '0.00'));
            $datiRiepilogo->appendChild($dom->createElement('Arrotondamento', '0.00'));

            $datiRiepilogo->appendChild($dom->createElement('ImponibileImporto', number_format($dati['imponibile'], 2)));
            //Se l'aliquota non è 0, bisogna specificare Imposta (2.2.1.14)
            if ($aliquota == 0) {
                $datiRiepilogo->appendChild($dom->createElement('Imposta','0.00'));
            } else {
                $datiRiepilogo->appendChild($dom->createElement('Imposta', number_format($dati['imposta'], 2)));
                $datiRiepilogo->appendChild($dom->createElement('EsigibilitaIVA', 'I'));
            }
        }
    }

    /**
     * Legge i campi extra del customer, Pec e SDI
     */
    protected function readExtraCustomerValues($id_customer = null)
    {
        if ($id_customer == null) {
            $id_customer = Context::getContext()->customer->id;
        }
        
        $query = 'SELECT `pec_email`, `sdi_code`'
            .' FROM `'. _DB_PREFIX_.'customer` c '
            .' WHERE c.id_customer = '.(int)$id_customer;
            
        return  Db::getInstance(_PS_USE_SQL_SLAVE_)->getRow($query);
    }

    private function getSoggettoEmittente($dom){
        return $dom->createElement('SoggettoEmittente', 'TZ');
    }

    private function getDatiTrasmissioneXML ($dom, $customerSdiCode, $customerPecEmail, $country) 
    {
        // Dati Trasmissione
        $datiTrasmissione = $dom->createElement('DatiTrasmissione');
        $idTrasmittente = $dom->createElement('IdTrasmittente');
        $datiTrasmissione->appendChild($idTrasmittente);
        $idTrasmittente->appendChild($dom->createElement('IdPaese', 'IT'));
        $idTrasmittente->appendChild($dom->createElement('IdCodice', '01589730629'));
        $datiTrasmissione->appendChild($dom->createElement('ProgressivoInvio', '0000000000'));
        $datiTrasmissione->appendChild($dom->createElement('FormatoTrasmissione', 'FPR12'));
        /**  Per B2C - Id Zone 7 San Marino 
        * N.B 
        * B2B non gestiti (occorre il CIF) 
        * PA non gestite (non esiste un modo per dedurlo se non facendolo specificare nel form di registrazione al cliente)
        **/

        if (isset($country->iso_code) && strcasecmp(trim($country->iso_code), 'SM') === 0) {
            //Codice Univoco B2C San Marino - 
                $datiTrasmissione->appendChild($dom->createElement('CodiceDestinatario', '2R4GT08'));
        }
        else if (!$customerSdiCode && isset($country->iso_code) && strcasecmp(trim($country->iso_code), 'IT') === 0) {
            $datiTrasmissione->appendChild($dom->createElement('CodiceDestinatario', '0000000')); 
        }
        else  if ($customerSdiCode && isset($country->iso_code) && strcasecmp(trim($country->iso_code), 'IT') === 0){
            $datiTrasmissione->appendChild($dom->createElement('CodiceDestinatario', $customerSdiCode));
        }
        else  if (isset($country->iso_code) && strcasecmp(trim($country->iso_code), 'IT') !== 0){
            $datiTrasmissione->appendChild($dom->createElement('CodiceDestinatario', 'XXXXXXX'));
        }
        if ($customerPecEmail) {
            $datiTrasmissione->appendChild($dom->createElement('PECDestinatario', $customerPecEmail));
        }
        
        return $datiTrasmissione;
    }

    private function getCedentePrestatoreXML($dom)
    {
        // Dati Anagrafici Cedente Prestatore
        $cedentePrestatore = $dom->createElement('CedentePrestatore');
        $datiAnagrafici = $dom->createElement('DatiAnagrafici');
        $cedentePrestatore->appendChild($datiAnagrafici);
        $idFiscaleIVA = $dom->createElement('IdFiscaleIVA');
        
        $datiAnagrafici->appendChild($idFiscaleIVA);
        $idFiscaleIVA->appendChild($dom->createElement('IdPaese', $this->nazioneCedente));
        $idFiscaleIVA->appendChild($dom->createElement('IdCodice', $this->cedenteIdFiscale));
        $datiAnagrafici->appendChild($dom->createElement('CodiceFiscale', $this->codiceFiscaleCedente));
        $anagrafica = $dom->createElement('Anagrafica');
        $datiAnagrafici->appendChild($anagrafica);
        $anagrafica->appendChild($dom->createElement('Denominazione', $this->denominazioneCedente));
        $datiAnagrafici->appendChild($dom->createElement('RegimeFiscale', $this->regimeFiscaleCedente ?? 'RF01'));
    
        // Sede Cedente Prestatore
        $sede = $dom->createElement('Sede');
        $cedentePrestatore->appendChild($sede);
        $sede->appendChild($dom->createElement('Indirizzo', $this->indirizzoCedente));
        $sede->appendChild($dom->createElement('NumeroCivico', $this->numeroCivicoCedente));
        $sede->appendChild($dom->createElement('CAP', $this->capCedente));
        $sede->appendChild($dom->createElement('Comune', $this->comuneCedente));
        $sede->appendChild($dom->createElement('Provincia', $this->provinciaCedente));
        $sede->appendChild($dom->createElement('Nazione', $this->nazioneCedente));
    
        // Contatti Cedente Prestatore
        $contatti = $dom->createElement('Contatti');
        $cedentePrestatore->appendChild($contatti);
        $contatti->appendChild($dom->createElement('Email', $this->emailCedente));
    
        // Riferimento Amministrazione
        if (isset($this->riferimentoAmministrazioneCedente) && strcasecmp(trim($this->riferimentoAmministrazioneCedente), '') !== 0){
            $cedentePrestatore->appendChild($dom->createElement('RiferimentoAmministrazione', $this->riferimentoAmministrazioneCedente));
        }

        return $cedentePrestatore;
        
    
    }

    private function getCessionarioCommittente($dom, $address, $customer, $country) {
        // Cessionario Committente
        $cessionarioCommittente = $dom->createElement('CessionarioCommittente');
        $datiAnagraficiCC = $dom->createElement('DatiAnagrafici');
        $cessionarioCommittente->appendChild($datiAnagraficiCC);
        $swDatiFiscali = false;
        if (isset($address->vat_number) && strcasecmp(trim($address->vat_number), '') !== 0) {
            $idFiscaleIVACC = $dom->createElement('IdFiscaleIVA');
            $datiAnagraficiCC->appendChild($idFiscaleIVACC);
            $idFiscaleIVACC->appendChild($dom->createElement('IdPaese', $country->iso_code));
            $idFiscaleIVACC->appendChild($dom->createElement('IdCodice', $address->vat_number));
            $swDatiFiscali = true;
        }

        if (strcasecmp(trim(strtoupper($country->iso_code)), 'IT') == 0 && isset($address->dni) && strcasecmp(trim($address->dni), '') !== 0) { 
            $datiAnagraficiCC->appendChild($dom->createElement('CodiceFiscale', strtoupper($address->dni)));
            $swDatiFiscali = true;
        }
        else if(strcasecmp(trim(strtoupper($country->iso_code)), 'IT') == 0 && isset($customer->siret) && strcasecmp(trim($customer->siret), '') !== 0) { 
            $datiAnagraficiCC->appendChild($dom->createElement('CodiceFiscale', strtoupper($customer->siret)));
            $swDatiFiscali = true;
        }


        // se non sono IT e non hanno nè codice fiscale e nè Vat Number imposto "ISOCODExxxxxxxxxx" es. USxxxxxxxxxxx
        if (strcasecmp(trim(strtoupper($country->iso_code)), 'IT') !== 0 && !$swDatiFiscali) {
            $idFiscaleIVACC = $dom->createElement('IdFiscaleIVA');
            $datiAnagraficiCC->appendChild($idFiscaleIVACC);
            $idFiscaleIVACC->appendChild($dom->createElement('IdPaese', $country->iso_code));
            $idFiscaleIVACC->appendChild($dom->createElement('IdCodice', 'XXXXXXXXXXX'));
        }
        
        $anagraficaCC = $dom->createElement('Anagrafica');
        $datiAnagraficiCC->appendChild($anagraficaCC);
        //??? */ se ha partita iva ha la denominazione altrimenti nome e cognome
        $anagraficaCC->appendChild($dom->createElement('Nome', $customer->firstname));
        $anagraficaCC->appendChild($dom->createElement('Cognome', $customer->lastname));
    
        //Stato
        $state = new State($address->id_state);
        $sedeCC = $dom->createElement('Sede');
        $cessionarioCommittente->appendChild($sedeCC);
        $sedeCC->appendChild($dom->createElement('Indirizzo', $address->address1 . " " . $address->address2));
        $sedeCC->appendChild($dom->createElement('CAP', $address->postcode));
        $sedeCC->appendChild($dom->createElement('Comune', $address->city));
        //dd($state, $address);
        if (Validate::isLoadedObject($state)) {
            
            $sedeCC->appendChild($dom->createElement('Provincia', strtoupper($state->iso_code)));
        }
        
                //dd($country);
        $sedeCC->appendChild($dom->createElement('Nazione', strtoupper($country->iso_code)));
        return $cessionarioCommittente;
    }

    private function getTerzoIntermediario($dom)
    {
        // Terzo Intermediario 
        $terzoIntermediario = $dom->createElement('TerzoIntermediarioOSoggettoEmittente');
        $datiAnagraficiTI = $dom->createElement('DatiAnagrafici');
        $terzoIntermediario->appendChild($datiAnagraficiTI);
        $idFiscaleIVATI = $dom->createElement('IdFiscaleIVA');
        $datiAnagraficiTI->appendChild($idFiscaleIVATI);
        $idFiscaleIVATI->appendChild($dom->createElement('IdPaese', 'IT'));
        $idFiscaleIVATI->appendChild($dom->createElement('IdCodice', '01589730629'));
        $datiAnagraficiTI->appendChild($dom->createElement('CodiceFiscale', '01589730629'));
        $anagraficaTI = $dom->createElement('Anagrafica');
        $datiAnagraficiTI->appendChild($anagraficaTI);
        $anagraficaTI->appendChild($dom->createElement('Denominazione', 'Cloud Finance S.r.l.'));
        
        return $terzoIntermediario;
    }

    public function downloadInvoiceXML($invoiceNumber, $xmlContent): void
    {

        // Imposta gli header per il download
        header('Content-Type: application/xml');
        header('Content-Disposition: attachment; filename="fattura_' . $invoiceNumber . '.xml"');
        header('Content-Length: ' . strlen($xmlContent));

        // Output dell'XML
        echo $xmlContent;

        // Interrompe l'esecuzione per garantire che nessun altro output venga incluso
        exit;
    }

    public function getOrderSlipNumberFormatted($orderSlip)
    {
        if (!Validate::isLoadedObject($orderSlip)) {
            return 'Errore: OrderSlip non valido';
        }
    
        $prefix = Configuration::get('PS_CREDIT_SLIP_PREFIX', Context::getContext()->language->id);
        if(!$prefix){
            return $orderSlip->id;
        }
        return sprintf('%1$s%2$06d', $prefix, (int) $orderSlip->id);
    }


    public function checkExistsInvoice($orderInvoice, $invoiceNumberFormatted){
        // Verifica se la fattura esiste già nel DB
        return Db::getInstance()->getValue(
            'SELECT id
             FROM `'. _DB_PREFIX_ .'freeinvoice_logs`
             WHERE invoice_number = "'. pSQL($invoiceNumberFormatted) .'"
               AND invoice_id != 0'
        );
    }

    /**
     * Invia la fattura tramite API.
     *
     * @param OrderInvoice $orderInvoice
     * @return array
     */
    public function sendInvoice($orderInvoice, $forced = false)
    {
        if (!$forced && !filter_var($this->automaticSyncCedente, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE)) { 
            $this->log->notice('Invio automatico disabilitato');
            return ['warning' => true, 'message' => 'Invio automatico disabilitato'];
        }
        
        if ($orderInvoice instanceof OrderInvoice) {
            $invoiceNumberFormatted = $orderInvoice->getInvoiceNumberFormatted(Context::getContext()->language->id);
            $isCreditNote = false;
        }
        
        if ($orderInvoice instanceof OrderSlip) {
            $invoiceNumberFormatted = $this->getOrderSlipNumberFormatted($orderInvoice);
            $isCreditNote = true;
        }
        
        if(!$forced && $this->checkExistsInvoice($orderInvoice, $invoiceNumberFormatted)){
            return ['warning' => true, 'message' => 'Fattura già inviata'];
        }
        
        $result = $this->generateInvoiceXML($orderInvoice);
        if (array_key_exists('type', $result) && $result['type'] === 'error') { 
            return ['error' => true, 'message' => $result['message']];
        }
        
        if (array_key_exists('type', $result) && $result['type'] === 'warning') { 
            return ['warning' => true, 'message' => $result['message']];
        }
        
        if (empty($result['xml'])) { 
            return ['error' => true, 'message' => 'Generazione XML Errore nella generazione del file XML. XML Vuoto'];
        }
        
        $xmlContent = $result['xml'];
        $xml = base64_encode($xmlContent);
        if (!$xml) {
            $this->log->error('Errore nella codifica Base64 del file XML');
            return ['error' => true, 'message' => 'Errore nella codifica Base64 del file XML'];
        }
        
        $response = $this->postRequest(
            $this->platformUrl, [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'apiKey' => $this->apiKey,
                'invoiceFileBase64' => $xml
            ], 
            $orderInvoice, 
            $isCreditNote
        );

        return $response; 
    }

    public function getInvoiceDetailsAndDownload($invoiceId)
    {
       
        $client = new \GuzzleHttp\Client();
        $apiGetInvoiceUrl = str_replace("{invoice_id}", $invoiceId, $this->apiGetInvoiceUrl);
        $this->log->notice("Download Invoice url: " . $this->apiGetInvoiceUrl );
        try {
            $response = $client->get($apiGetInvoiceUrl, [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'query' => [
                    'apiKey' => $this->apiKey,
                    'withFile' => 'true'
                ],
            ]);
            $responseBody = $response->getBody()->getContents();
            $responseData = json_decode($responseBody, true);
            if(empty($responseData) || !array_key_exists('data', $responseData) || empty($responseData['data'])) {
                return [
                    'error' => true,
                    'message' => "Errore nella chiamata API"
                ];
            }
            $response = $responseData['data'];
            $invoiceNumber = $response['invoiceBody']['number'];
            $this->log->notice('API Response: ' . print_r($responseData, true));
            if (isset($response['invoiceFileBase64']) && !empty($response['invoiceFileBase64'])) {
                $xmlContent = base64_decode($response['invoiceFileBase64']);
                $this->downloadInvoiceXML($invoiceNumber, $xmlContent);
            } 
            else 
            if (isset($response['invoiceFileXmlBase64']) && !empty($response['invoiceFileXmlBase64'])) {
                $xmlContent = base64_decode($response['invoiceFileXmlBase64']);
                $this->downloadInvoiceXML($invoiceNumber, $xmlContent);
            }
            else {
                return [
                    'error' => true,
                    'message' => "InvoiceFileXml non presente nella risposta API"
                ];
            }

            return [
                'success' => true,
                'message' => $response
            ];
        } catch (\GuzzleHttp\Exception\ConnectException $e) {
            $errorMessage = "Impossibile raggiungere l'host.";
            $this->log->error($errorMessage . ' Dettagli: ' . $e->getMessage());

            return [
                'error' => true,
                'message' => $errorMessage
            ];
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            $statusCode = $e->getResponse() ? $e->getResponse()->getStatusCode() : 'N/A';
            $errorMessage = "Errore HTTP $statusCode: " . $e->getMessage();
            $this->log->error($errorMessage);

            return [
                'error' => true,
                'message' => $errorMessage
            ];
        } catch (\Exception $e) {
            $errorMessage = "Errore sconosciuto nell'API: " . $e->getMessage();
            $this->log->error($errorMessage);

            return [
                'error' => true,
                'message' => $errorMessage
            ];
        }
    }


    /**
     * Esegue una richiesta POST asincrona.
     *
     * @param string $url
     * @param array $data
     * @return void
     */
    private function postRequest(string $url, array $data, $orderInvoice, $isCreditNote)
    {
        $client = new \GuzzleHttp\Client();
    
        try {
            $response = $client->post($url, [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'body' => json_encode($data),
            ]);
    
            $responseBody = $response->getBody()->getContents();
            $responseData = json_decode($responseBody, true);
            $errors = "";
            $this->log->notice('API Response: ' . print_r($responseData, true));
            if (array_key_exists('errors', $responseData) && empty($responseData['data'])) {
                //
                foreach ($responseData['errors'] as $error) {
                    $this->log->error($error);
                    $errors .= $error['message'];
                }
                $this->logInvoiceResponse([], $orderInvoice, $isCreditNote);
                return [
                    'error' => true,
                    'message' => $errors
                ];
            }
            
            $this->logInvoiceResponse($responseData, $orderInvoice, $isCreditNote);
            if ($isCreditNote){
                return [
                    'success' => true,
                    'message' => "Nota di Credito inviata correttamente"
                ];
            }

            return [
                'success' => true,
                'message' => "Fattura inviata correttamente"
            ];
        } catch (\GuzzleHttp\Exception\ConnectException $e) {
            $errorMessage = "Impossibile raggiungere l'host.";
            $this->log->error($errorMessage . ' Dettagli: ' . $e->getMessage());
    
            return [
                'error' => true,
                'message' => $errorMessage
            ];
        } catch (\GuzzleHttp\Exception\RequestException $e) {
            // Errore HTTP (es. 500, 403, 404)
            $statusCode = $e->getResponse() ? $e->getResponse()->getStatusCode() : 'N/A';
            $errorMessage = "Errore HTTP $statusCode: " . $e->getMessage();
            $this->log->error($errorMessage);
    
            return [
                'error' => true,
                'message' => $errorMessage
            ];
        } catch (\Exception $e) {
            // Errore generico
            $errorMessage = "Errore sconosciuto nell'API: " . $e->getMessage();
            $this->log->error($errorMessage);
    
            return [
                'error' => true,
                'message' => $errorMessage
            ];
        }
    }
    

    private function logInvoiceResponse(array $responseData, $orderInvoice, $isCreditNote): void
    {
         
        $type = $isCreditNote ? 'slip' : 'invoice';
        
        if (!isset($responseData['data']) || empty($responseData['data'])) {
            $this->log->error('Risposta API non valida.');
            $statusName = "Invio Fallito";
            if (!$isCreditNote) { 
                $invoiceNumber = $orderInvoice->getInvoiceNumberFormatted(Context::getContext()->language->id);
            }
            else {
                $invoiceNumber = $this->getOrderSlipNumberFormatted($orderInvoice);
            }
            Db::getInstance()->insert('freeinvoice_logs', [
                'ps_order_id' =>$orderInvoice->id_order,
                'ps_order_invoice_id' => $orderInvoice->id,
                'invoice_number' => $invoiceNumber,
                'status_name' => $statusName,
                'type' => $type,
                'date_add' => date('Y-m-d H:i:s'),
                'date_upd' => date('Y-m-d H:i:s'),
            ]);
            
            $this->log->notice('Nuova '. $type .'  registrata - Stato: ' . $statusName);
            return;
        }

         // Inserisce un nuovo record
        

        foreach ($responseData['data'] as $invoice) {
            $invoiceId = (int)($invoice['invoiceId'] ?? 0);
            $statusId = (int)($invoice['invoiceStatus'] ?? 0);
            $statusName = pSQL($invoice['invoiceStatusName'] ?? 'Sconosciuto');
            $invoiceNumber = pSQL($invoice['invoiceBody']['number'] ?? 'N/A');
            $totalAmount = (float)($invoice['invoiceBody']['totalAmount'] ?? 0.00);
            $dateAdded = pSQL($invoice['timestamp'] ?? date('Y-m-d H:i:s'));
            $xmlPath = pSQL($invoice['invoiceFileName'] ?? '');

        }
        // Inserisce un nuovo record
        Db::getInstance()->insert('freeinvoice_logs', [
            'invoice_id' => $invoiceId,
            'ps_order_id' => $orderInvoice->id_order,
            'ps_order_invoice_id' => $orderInvoice->id,
            'invoice_number' => $invoiceNumber,
            'status_id' => $statusId,
            'status_name' => $statusName,
            'type' => $type,
            'date_add' => $dateAdded,
            'date_upd' => date('Y-m-d H:i:s'),
        ]);
        $this->log->notice('Nuova fattura registrata: ' . $invoiceNumber . ' - Stato: ' . $statusName);
    }



}
