<?php

declare(strict_types=1);
use FreeInvoice\Services\NotificationService;
use FreeInvoice\Services\FreeInvoiceService;
use FreeInvoice\Services\CloudFinanceLogger;
class AdminFreeInvoiceController extends ModuleAdminController
{
    private $log = null;
    public function __construct()
    {
        parent::__construct();
        $this->bootstrap = true;
        $this->table = 'freeinvoice_logs';
        $this->className = 'freeinvoiceLog';
        $this->identifier = 'id';
        $this->name = 'freeinvoice';
        $this->version = '1.0.0';
        $this->author = 'Giuseppe Paglialonga';
        $this->templatePath = _PS_MODULE_DIR_ . $this->name . '/views/templates/admin/';
        $this->log = CloudFinanceLogger::create();
        $this->fields_list = [
            'ps_order_id' => [
                'title' => $this->l('ID Ordine PS'),
                'type' => 'int',
                'align' => 'center',
                'callback' => 'renderOrderLink',
                'search' => true,
                'orderby' => true,
            ],
            'invoice_id' => [
                'title' => $this->l('ID Fattura'),
                'type' => 'int',
                'align' => 'center',
                'search' => true,
                'orderby' => true,
            ],
            'invoice_number' => [
                'title' => $this->l('Numero Fattura'),
                'type' => 'text',
                'align' => 'center',
                'search' => true,
                'orderby' => true,
            ],
            'status_name' => [
                'title' => $this->l('Stato'),
                'type' => 'text',
                'search' => true,
                'orderby' => true,
            ],
            'date_add' => [
                'title' => $this->l('Data Invio'),
                'type' => 'datetime',
                'search' => true,
                'orderby' => true,
            ],
            'date_upd' => [
                'title' => $this->l('Data Aggiornamento'),
                'type' => 'datetime',
                'search' => true,
                'orderby' => true,
            ],
            'actions' => [
                'title' => $this->l('Azioni'),
                'type' => 'text',
                'align' => 'center',
                'callback' => 'renderActionsButtons',
                'orderby' => false,
                'search' => false,
            ],
        ];
    }
    public function setMedia($isNewTheme = false): void
    {
        parent::setMedia($isNewTheme);
        $this->addJS(_PS_MODULE_DIR_ . 'freeinvoice/views/js/notifications.js');
        $this->addCSS(_PS_MODULE_DIR_ . 'freeinvoice/views/css/notifications.css');
    }
    public function initContent(): void
    {
        parent::initContent();
        $this->content = $this->renderList();
        $this->context->smarty->assign(['content' => $this->content]);
    }

    public function renderOrderLink($orderId, $row)
    {
        $link = $this->context->link->getAdminLink('AdminOrders', true, [], ['vieworder' => 1, 'id_order' => (int) $orderId]);

        return '<a href="' . htmlspecialchars($link) . '" target="_blank">#' . (int) $orderId . '</a>';
    }

    public function renderList(): string
    {
        $helper = new HelperList();
        $helper->shopLinkType = '';
        $helper->simple_header = false;
        $helper->identifier = 'id';
        $helper->table = 'freeinvoice_logs';
        $helper->no_link = true;
        $helper->title = $this->l('Log delle Fatture');
        $helper->currentIndex = $this->context->link->getAdminLink('AdminFreeInvoice', false);
        $helper->token = Tools::getAdminTokenLite('AdminFreeInvoice');

        $helper->list_id = 'freeinvoice_logs';
        $helper->module = $this->module;

        $where = '1=1';

        // Filtri di ricerca
        if ($orderId = Tools::getValue('freeinvoice_logsFilter_ps_order_id')) {
            $where .= ' AND ps_order_id = ' . (int) $orderId;
        }

        if ($invoiceId = Tools::getValue('freeinvoice_logsFilter_invoice_id')) {
            $where .= ' AND invoice_id = ' . (int) $invoiceId;
        }

        if ($invoiceNumber = Tools::getValue('freeinvoice_logsFilter_invoice_number')) {
            $where .= ' AND invoice_number LIKE "%' . pSQL($invoiceNumber) . '%"';
        }

        if ($status = Tools::getValue('freeinvoice_logsFilter_status_name')) {
            $where .= ' AND status_name LIKE "%' . pSQL($status) . '%"';
        }

        $dateAddValues = Tools::getValue('freeinvoice_logsFilter_date_add', []);
        if (is_array($dateAddValues) && isset($dateAddValues[0]) && !empty($dateAddValues[0])) {
            $where .= ' AND date_add >= "' . pSQL($dateAddValues[0]) . '"';
        }
        if (is_array($dateAddValues) && isset($dateAddValues[1]) && !empty($dateAddValues[1])) {
            $where .= ' AND date_add <= "' . pSQL($dateAddValues[1]) . '"';
        }

        $dateUpdValues = Tools::getValue('freeinvoice_logsFilter_date_upd', []);
        if (is_array($dateUpdValues) && isset($dateUpdValues[0]) && !empty($dateUpdValues[0])) {
            $where .= ' AND date_upd >= "' . pSQL($dateUpdValues[0]) . '"';
        }
        if (is_array($dateUpdValues) && isset($dateUpdValues[1]) && !empty($dateUpdValues[1])) {
            $where .= ' AND date_upd <= "' . pSQL($dateUpdValues[1]) . '"';
        }

        // **GESTIONE ORDER BY DINAMICO**
        $defaultOrderBy = 'id'; // Campo di default per l'ordinamento
        $defaultOrderWay = 'DESC'; // Ordine predefinito

        $orderBy = Tools::getValue($helper->list_id . 'Orderby', $defaultOrderBy);
        $orderWay = Tools::getValue($helper->list_id . 'Orderway', $defaultOrderWay);

        // Evita SQL Injection sui campi non numerici
        $validOrderByFields = ['id', 'ps_order_id', 'invoice_id', 'invoice_number', 'status_name', 'date_add', 'date_upd'];
        if (!in_array($orderBy, $validOrderByFields)) {
            $orderBy = $defaultOrderBy;
        }

        $orderWay = strtoupper($orderWay);
        if (!in_array($orderWay, ['ASC', 'DESC'])) {
            $orderWay = $defaultOrderWay;
        }

        // **APPLICA L'ORDINAMENTO ALLA QUERY**
        $sql = 'SELECT *, "" as actions FROM `' . _DB_PREFIX_ . 'freeinvoice_logs`
                WHERE ' . $where . '
                ORDER BY `' . bqSQL($orderBy) . '` ' . pSQL($orderWay);

        $logs = Db::getInstance()->executeS($sql);

        return $helper->generateList($logs, $this->fields_list);
    }


    public function renderActionsButtons($value, $row)
    {   
        $invoiceId = isset($row['invoice_id']) ? (int) $row['invoice_id'] : 0;
        $invoiceOrderId = isset($row['ps_order_invoice_id']) ? (int) $row['ps_order_invoice_id'] : 0;

        $baseUrl = $this->context->link->getAdminLink('AdminFreeInvoice', true);
        $downloadLink = $baseUrl . '&downloadInvoiceXML=1&invoice_id=' . $invoiceId . '&ps_invoice_order_id=' . $invoiceOrderId . '&type=' . $row['type'] . '&id=' . $row[ 'id'];
        $resendLink = $baseUrl . '&resendInvoice=1&invoice_id=' . $invoiceId . '&type=' . $row['type']. '&id=' . $row[ 'id'];

        return '<div class="btn-group">
                    <a href="' . $downloadLink . '" class="btn btn-default btn-sm"><i class="icon-download"></i> Scarica XML</a>
                    <a href="' . $resendLink . '" class="btn btn-warning btn-sm"><i class="icon-mail-forward"></i> Reinvia</a>
                </div>';
    }

    public function postProcess()
    {   
        if (isset($this->context->cookie->growl_success)) {
            NotificationService::success($this->context->cookie->growl_success);

            unset($this->context->cookie->growl_success);
        }
        if (isset($this->context->cookie->growl_error)) {
            NotificationService::error($this->context->cookie->growl_error);

            unset($this->context->cookie->growl_error);
        }
        
        $downloadInvoiceXmlSubmit = Tools::isSubmit('downloadInvoiceXML');
        $resendInvoiceSubmit = Tools::isSubmit('resendInvoice');
        if (!$downloadInvoiceXmlSubmit && !$resendInvoiceSubmit) {
            return;
        }
        
        $service = new FreeInvoiceService($this->log);
        $invoiceId = (int) Tools::getValue('invoice_id');
        $id = (int) Tools::getValue('id');
        $type = Tools::getValue('type');
        $messageType = "";
        $existingLog = Db::getInstance()->getRow(
            'SELECT invoice_id, ps_order_invoice_id, status_id FROM `' . _DB_PREFIX_ . 'freeinvoice_logs` 
            WHERE id = "' . pSQL($id) . '"'
        );
        $psOrderInvoiceId = $existingLog['ps_order_invoice_id'];
        $isCreditNote = null;
        if ($type === 'slip'){
            $isCreditNote = true;
            $messageType = 'Nota di Credito';
            $orderInvoice = new OrderSlip((int) $psOrderInvoiceId);
            $invoiceNumber = $service->getOrderSlipNumberFormatted($orderInvoice);
        }

        if($type === 'invoice'){
            $isCreditNote = false;
            $messageType = 'Fattura';
            $orderInvoice = new OrderInvoice($psOrderInvoiceId);
            $invoiceNumber = $orderInvoice->getInvoiceNumberFormatted(Context::getContext()->language->id);
        }
        $this->log->debug('PostProcess: ' .  $messageType);

        

        if (!Validate::isLoadedObject($orderInvoice)) {
            NotificationService::error('Errore ' . $messageType . ' non valida. Contattare l\'assistenza.');
            return;  
        }
        if ($downloadInvoiceXmlSubmit) {
            
            $statusId = $existingLog['status_id'];
            if ($statusId == 8 || $statusId == 9) {

                $result = $service->getInvoiceDetailsAndDownload($existingLog[ 'invoice_id']);
                if (array_key_exists('error', $result) && $result['error']) { 
                    NotificationService::error($messageType . ' N. ' . $invoiceNumber . ': ' . $result['message']);
                    return;
                }
            }

            $result = $service->generateInvoiceXML($orderInvoice);
            if (array_key_exists('type', $result) && $result['type'] === 'error') { 
                
                NotificationService::error($result['message']);
            }

            if (empty($result['xml'])) { 
                $this->context->cookie->growl_error = 'Generazione XML Errore nella generazione del file XML. XML Vuoto. Contattare l\'assistenza.';
            }
            $xmlContent = $result['xml'];
            $service->downloadInvoiceXML($invoiceNumber, $xmlContent);
            Tools::redirectAdmin($this->context->link->getAdminLink('AdminFreeInvoice'));
        }
        
        if ($resendInvoiceSubmit) {
            $result = $service->sendInvoice($orderInvoice, true);
            if (array_key_exists('success', $result) && $result['success']) {
                $this->context->cookie->growl_success = $messageType . ' N. ' . $invoiceNumber . ' inviata con successo!';
            } 

            if ((array_key_exists('error', $result) && $result['error']) || (array_key_exists('warning', $result) && $result['warning'])) {
                $this->context->cookie->growl_error = 'Errore Invio ' . $messageType . ' N. ' . $invoiceNumber . ': '. $result['message'];
            }
            
        }
        Tools::redirectAdmin($this->context->link->getAdminLink('AdminFreeInvoice'));
    }

}
